<?php

// Diese Klasse stellt Funktionen für eine kontextbezogene Datenbankadministration bereit.
//
// Ein "Kontext" ist ein Namenspräfix, den sich mehrere zusammengehörige Tabellen teilen.
// Mit Hilfe eines Kontextes kann man auf einfache Weise Aktionen ausführen, die mehrere Tabellen 
// gleichzeitig betreffen, zum Beispiel Umbenennen, Löschen, Sichern, Wiederherstellen.
//
// Ein wichtiger Aspekt bei der kontextbezogenen Verarbeitung besteht darin, dass die 
// Konsistenz der aufeinander bezogenen Tabellen sichergestellt bleibt, auch wenn z. B.  beim 
// Einlesen eines Dumps Fehler auftreten.


// TODO: Engines, Charsets etc. in Backup-Datei speichern und beim Restore berücksichtigen


class Database_Administrator {
	
	var $context = '';
	var $db; // the mysqli object for communicating with the databse
	var $tables = array();
	var $schemas_mandatory = array();
	var $schemas_default = array();
	var $schemaMapRules = array();
	var $backupPrefix;
	var $insertBuffer = array();
	var $lastErrno;
	var $lastError;
	var $lastQuery;

	
	function __construct( $context, $db )
	{
		$this->context = $context;
		$this->db = $db;
		$this->tables = $this->get_tables();
	}

	
	function dump( $file, $exclude = array() )
	{
		$this->lastErrno = NULL;
		$this->lastError = NULL;

		if ( empty( $this->tables ) )
		{
			$this->lastErrno = 1;
			$this->lastError = 'There are no tables to back-up';
			return FALSE;
		}

		if ( strpos( $file, '/' ) === FALSE )
		{
			if ( ob_get_status() )
			{
				ob_end_clean();
			}
			
			$extension = substr( strrchr( $file, '.' ), 1 );
			header( 'Content-Type: application/' . ( empty( $extension ) ? 'octetstream' : $extension ) );
			header( 'Content-Disposition: attachment; filename="' . $file . '"' );
			header( 'Pragma: no-cache' );
			$fp = FALSE;
		}
		else
		{
			$fp = fopen( $file, 'w' );

			if ( ! $fp )
			{
				$this->lastErrno = 2;
				$this->lastError = 'Could not open backup file for writing';
				return FALSE;
			}
		}

		if ( substr( $file, -4 ) == '.php' )
		{
			$this->_dump_line( $fp, '<?php die(\'Access denied\'); ?'.'>' );
		}

		$this->_dump_line( $fp, '#version: 1.1'  );
		$this->_dump_line( $fp, '#date: ' . date( 'Y-m-d H:i' ) );
		$this->_dump_line( $fp, "#context: {$this->context}" );
		$this->_dump_line( $fp );

		foreach ( $this->tables as $table )
		{
			$table_name = substr( $table, strlen( $this->context ) );

			if ( in_array( $table_name, $exclude ) )
			{
				continue;
			}

			$selection = $this->db->query( "SELECT * FROM `{$table}`" );

			if ( $selection->num_rows == 0 )
			{
				continue;
			}

			$this->_dump_line( $fp, "#table: {$table_name}" );
			$result = $this->db->query( "SHOW CREATE TABLE `{$table}`" );
			$row = $result->fetch_row();
			$schema = substr( $row[1], strpos( $row[1], '(' ) + 1, strrpos( $row[1], ')' ) - strlen( $row[1] ) );
			$this->_dump_line( $fp, '#schema: ' . str_replace( '   ', ' ', str_replace( "\n", ' ', $schema ) ) );
			$fields = array_keys( $selection->fetch_assoc() );
			$this->_dump_line( $fp, '#fields: `' . implode( '`,`', $fields ) . '`' );
			$selection->data_seek( 0 );

			while ( $row = $selection->fetch_row() )
			{
				$values = array();

				foreach ( $row as $value )
				{
					$values[] = is_null( $value ) ? 'NULL' : "'" . $this->db->real_escape_string( $value ) . "'";
				}

				$this->_dump_line( $fp, implode( ',', $values ) );
			}

			$this->_dump_line( $fp );
		}

		if ( ! $fp )
		{
			exit;
		}

		fclose( $fp );
		return TRUE ;
	}


	function import_dump( $file, $safeMode = TRUE, $keepBackup = FALSE )
	{
		$this->lastErrno = NULL;
		$this->lastError = NULL;
		$fp = fopen( $file, 'r' );

		if ( ! $fp )
		{
			$this->lastErrno = 1;
			$this->lastError = "Could not open file {$file} for reading";
			return FALSE;
		}

		// Rename or empty existing tables

		if ( $this->backup_context( $keepBackup ) )
		{
			$this->tables = array();
		}
		elseif ( $safeMode )
		{
			$this->lastErrno = 2;
			$this->lastError = 'Could not backup database tables';
			return FALSE;
		}
		else
		{
			$this->empty_context();
		}

		// Import dump file

		if ( substr( $file, -4 ) == '.php' )
		{
			fgets( $fp ); // erste Zeile (PHP-Wrapper) überspringen
		}

		$table = '';
		$lineCounter = 0;

		while ( ( $line = $this->_read_line_from_dump( $fp ) ) !== FALSE )
		{
			$lineCounter++;

			// Handle begin of new table

			if ( substr( $line, 0, 7 ) == '#table:' )
			{
				$table = trim( substr( $line, 7 ) );
				$schema = '';
				$fields = '';
				$line = $this->_read_line_from_dump( $fp );
				$lineCounter++;

				if ( substr( $line, 0, 8 )  == '#schema:' )
				{
					$schema = substr( $line, 8 );
					$line = $this->_read_line_from_dump( $fp );
					$lineCounter++;
				}

				if ( ! $this->_prepare_table_for_import( $table, $schema ) )
				{
					$this->lastErrno = 4;
					$this->lastError = "Preparation of table $table failed";
					break;
				}

				if ( substr( $line, 0, 8 )  == '#fields:' )
				{
					$fields = substr( $line, 8 );
					$line = $this->_read_line_from_dump( $fp );
					$lineCounter++;
				}
			}
			
			// Handle records

			if ( ! empty( $table ) && ! empty( $line ) && $line[0] == "'" )
			{
				if ( ! $this->_insert_via_buffer( $table, $fields, $line ) )
				{
					$this->lastErrno = 3;
					$this->lastError = "Error processing input buffer";
					break;
				}
			}
		}

		// Finish
				
		$this->_process_insert_buffer();

		if ( ! $keepBackup )
		{
			$this->drop_context( $this->backupPrefix );
		}

		fclose( $fp );
		return $this->lastErrno == NULL;
	}
	
	
	function _insert_via_buffer( $table, $fields, $values )
	{
		if ( ! empty( $this->insertBuffer['table'] ) 
				&& $table != $this->insertBuffer['table'] )
		{
			if ( ! $this->_process_insert_buffer() )
			{
				return FALSE;
			}
		}
		
		if ( empty( $this->insertBuffer['table'] ) )
		{
			$this->insertBuffer['table'] = $table;
			$this->insertBuffer['fields'] = $fields;
			$this->insertBuffer['records'] = array();
			$this->insertBuffer['size'] = 0;
		}

		$this->insertBuffer['records'][] = $values;
		$this->insertBuffer['size'] += strlen( $values );
		
		if ( $this->insertBuffer['size'] > 250 * 1024 )
		{
			return $this->_process_insert_buffer();
		}
		
		return TRUE;
	}
	

	function _process_insert_buffer()
	{
		if ( empty( $this->insertBuffer['records'] ) )
		{
			return TRUE;
		}

		$records = implode( '), (', $this->insertBuffer['records'] );
		$query = "INSERT INTO `{$this->context}{$this->insertBuffer['table']}` ( {$this->insertBuffer['fields']} ) VALUES ( {$records} )";
		$this->lastQuery = $query;
		$result = $this->db->query( $query );

		if ( ! $result )
		{
			return FALSE;
		}
		
		$this->insertBuffer = array();
		return TRUE;
	}
	
	
	function drop_context( $prefix = '' )
	{
		$prefix = empty( $prefix ) ? $this->context : $prefix;
		$result = $this->db->query( "SHOW TABLES LIKE '{$prefix}%'" );
		
		if ( $result )
		{
			while ( $row = $result->fetch_row() )
			{
				if ( ! $this->db->query( "DROP TABLE `{$row[0]}`" ) )
				{
					$this->lastErrno = 1;
					$this->lastError = "Table {$row[0]} could not be dropped";
					return FALSE;
				}
			}
		}
		
		return TRUE;
	}
	

	function empty_context( $prefix = '' )
	{
		$prefix = empty( $prefix ) ? $this->context : $prefix;
		$result = $this->db->query( "SHOW TABLES LIKE '{$prefix}%'" );
		
		if ( $result )
		{
			while ( list( $table ) = $result->fetch_row() )
			{
				$this->db->query( "DELETE FROM `{$table}` WHERE 1" );
				$statements = array();
				$selection = $this->db->query( "SHOW COLUMNS FROM `{$table}`" );

				while ( list( $column ) = $selection->fetch_row() )
				{
					$statements[] = "DROP COLUMN `{$column}`";
				}

				$keylist = array();
				$selection = $this->db->query( "SHOW INDEX FROM `{$table}`" );

				while ( $row = $selection->fetch_assoc() )
				{
					if ( ! in_array( $row['Key_name'], $keylist ) )
					{
						$statements[] = "DROP INDEX `{$row['Key_name']}`";
						$keylist[] = $row['Key_name'];
					}
				}

				if ( ! empty( $statements ) )
				{
					$this->db->query( "ALTER TABLE `{$table}` ADD `vtdummy` int(1) default '0', " . implode( ', ', $statements ) );
				}
			}
		}
	}


	function backup_context( $keepBackup = FALSE )
	{
		if ( ! empty( $this->backupPrefix ) )
		{
			return FALSE;
		}

		do
		{
			$this->backupPrefix = '_backup_' . date( 'Ymdhi' ) . '_';
			sleep( 1 );
		}
		while ( $this->get_tables( $this->backupPrefix ) != array() );
		
		if ( ! $keepBackup )
		{
			register_shutdown_function( array( $this, 'restore_context' ) );
		}

		return $this->rename_context( $this->context, $this->backupPrefix );
	}


	function restore_context()
	{
		if ( empty( $this->backupPrefix ) )
		{
			return FALSE;
		}

		if ( ! $this->rename_context( $this->backupPrefix, $this->context ) )
		{
			return FALSE;
		}
		
		$this->backupPrefix = '';
		return TRUE;
	}


	function rename_context( $oldPrefix, $newPrefix )
	{
		if ( $oldPrefix == $newPrefix )
		{
			return TRUE;
		}

		$tables = $this->get_tables( $oldPrefix );
		$renames = array();

		foreach ( $tables as $table )
		{
			$renames[] = "`{$table}` TO `{$newPrefix}" . substr( $table, strlen( $oldPrefix ) ) . '`';
		}

		if ( empty( $renames ) )
		{
			return TRUE;
		}

		return $this->db->query( 'RENAME TABLE ' . implode( ', ', $renames ) );
	}


	function get_tables( $prefix = '' )
	{
		$prefix = empty( $prefix ) ? $this->context : $prefix;
		$result = $this->db->query( "SHOW TABLES" . ( empty( $prefix ) ? '' : " LIKE '{$prefix}%'" ) );
		$tables = array();
		
		if ( $result )
		{
			while ( $row = $result->fetch_row() )
			{
				$tables[] = $row[0];
			}
		}
		
		return $tables;
	}


	function _dump_line( $fp, $line = '' )
	{
		if ( $fp )
		{
			fwrite( $fp, $line . "\n" );
		}
		else
		{
			echo $line . "\n";
		}
	}


	function _read_line_from_dump( $fp )
	{
		if ( ! $line = fgets( $fp ) )
		{
			return FALSE;
		}

		return substr( $line, 0, -1 );
	}


	function _prepare_table_for_import( $table, $schema )
	{
		$key = preg_replace( array_keys( $this->schemaMapRules ), array_values( $this->schemaMapRules ), $table );

		if ( ! empty( $this->schemas_mandatory[ $key ] ) )
		{
			$schema = $this->schemas_mandatory[ $key ];
		}
		elseif ( empty( $schema ) && ! empty( $this->schemas_default[ $key ] ) )
		{
			$schema = $this->schemas_default[ $key ];
		}
		elseif ( empty( $schema ) )
		{
			return FALSE;
		}

		if ( in_array( $this->context . $table, $this->tables ) )
		{
			$schema = explode( ',', $schema );
			$this->lastQuery = "ALTER TABLE `{$this->context}{$table}` DROP COLUMN `vtdummy`, ADD " . implode( ', ADD ', $schema );
			return $this->db->query( $this->lastQuery );
		}

		$this->lastQuery = "CREATE TABLE `{$this->context}{$table}` ( {$schema} ) ENGINE='MyISAM' CHARACTER SET 'utf8mb4' COLLATE 'utf8mb4_unicode_ci'";
		return $this->db->query( $this->lastQuery );
	}

}

// end of file
